import { useState, useEffect } from 'react'
import { motion } from 'framer-motion'
import { Music2, Calendar, MessageSquare, Plus, Radio, Disc, Volume2 } from 'lucide-react'
import { useNavigate } from 'react-router-dom'
import ImageSlider from '../components/ImageSlider'
import './MyRentals.css'

const MyRentals = () => {
  const navigate = useNavigate()
  const [rentals, setRentals] = useState([])

  useEffect(() => {
    const saved = localStorage.getItem('musicRentals')
    if (saved) {
      try {
        setRentals(JSON.parse(saved))
      } catch (e) {
        setRentals([])
      }
    } else {
      const defaultRentals = [
        {
          id: 1,
          equipment: 'Fender Stratocaster',
          date: '2025-01-25',
          review: 'Отличная гитара, звук превосходный!',
          photo: '/images/rental-1.jpg'
        },
        {
          id: 2,
          equipment: 'Yamaha P-125',
          date: '2025-01-20',
          review: 'Клавиши очень отзывчивые, рекомендую',
          photo: '/images/rental-2.jpg'
        },
        {
          id: 3,
          equipment: 'Shure SM58',
          date: '2025-01-15',
          review: 'Качественный микрофон, без нареканий',
          photo: '/images/rental-3.jpg'
        }
      ]
      setRentals(defaultRentals)
      localStorage.setItem('musicRentals', JSON.stringify(defaultRentals))
    }
  }, [])

  return (
    <div className="my-rentals-wave">
      <div className="wave-background">
        <div className="wave wave-1"></div>
        <div className="wave wave-2"></div>
        <div className="wave wave-3"></div>
      </div>
      
      <div className="container">
        <motion.section 
          className="hero-wave-section"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ duration: 0.8 }}
        >
          <div className="hero-content-wave">
            <motion.div 
              className="hero-badge-wave"
              initial={{ scale: 0 }}
              animate={{ scale: 1 }}
              transition={{ delay: 0.2, type: "spring" }}
            >
              <Volume2 size={16} strokeWidth={2.5} />
              <span>АКТИВНЫЕ АРЕНДЫ</span>
            </motion.div>
            
            <motion.h1 
              className="hero-title-wave"
              initial={{ y: 30, opacity: 0 }}
              animate={{ y: 0, opacity: 1 }}
              transition={{ delay: 0.3 }}
            >
              Мои <span className="title-gradient-wave">аренды</span>
            </motion.h1>
            
            <motion.p 
              className="hero-description-wave"
              initial={{ y: 20, opacity: 0 }}
              animate={{ y: 0, opacity: 1 }}
              transition={{ delay: 0.4 }}
            >
              Управляйте арендованным оборудованием и отслеживайте сроки
            </motion.p>
            
            <motion.button 
              className="cta-wave-button"
              onClick={() => navigate('/rent')}
              initial={{ y: 20, opacity: 0 }}
              animate={{ y: 0, opacity: 1 }}
              transition={{ delay: 0.5 }}
              whileHover={{ scale: 1.05 }}
              whileTap={{ scale: 0.95 }}
            >
              <Plus size={20} strokeWidth={2.5} />
              <span>Новая аренда</span>
            </motion.button>
          </div>
          
          <motion.div 
            className="hero-visual-wave"
            initial={{ x: 100, opacity: 0 }}
            animate={{ x: 0, opacity: 1 }}
            transition={{ delay: 0.6 }}
          >
            <div className="sound-visualizer">
              {[...Array(12)].map((_, i) => (
                <motion.div
                  key={i}
                  className="sound-bar"
                  animate={{
                    height: [20, 60, 30, 50, 40, 70, 25, 55, 35, 45],
                  }}
                  transition={{
                    duration: 1.5,
                    repeat: Infinity,
                    delay: i * 0.1,
                    ease: "easeInOut"
                  }}
                />
              ))}
            </div>
          </motion.div>
        </motion.section>

        <ImageSlider />

        <section className="rentals-masonry-section">
          {rentals.length === 0 ? (
            <motion.div 
              className="empty-state-wave"
              initial={{ opacity: 0, scale: 0.9 }}
              animate={{ opacity: 1, scale: 1 }}
            >
              <div className="empty-icon-container">
                <Radio size={64} strokeWidth={1.5} />
                <div className="empty-pulse"></div>
              </div>
              <h2 className="empty-title-wave">Нет активных аренд</h2>
              <p className="empty-text-wave">Начните с аренды первого инструмента</p>
              <button className="empty-button-wave" onClick={() => navigate('/rent')}>
                Арендовать
              </button>
            </motion.div>
          ) : (
            <div className="masonry-grid-wave">
              {rentals.map((rental, index) => {
                const date = new Date(rental.date).toLocaleDateString('ru-RU', {
                  day: 'numeric',
                  month: 'long',
                  year: 'numeric'
                })

                return (
                  <motion.div
                    key={rental.id}
                    className={`rental-masonry-card card-${index % 3}`}
                    initial={{ opacity: 0, y: 50 }}
                    animate={{ opacity: 1, y: 0 }}
                    transition={{ delay: 0.2 + index * 0.15 }}
                    whileHover={{ y: -10 }}
                  >
                    <div className="card-image-masonry">
                      <img src={rental.photo} alt={rental.equipment} />
                      <div className="card-overlay-masonry">
                        <div className="card-status-masonry">
                          <Disc size={14} strokeWidth={2} />
                          <span>Активна</span>
                        </div>
                      </div>
                    </div>
                    <div className="card-content-masonry">
                      <h3 className="card-title-masonry">{rental.equipment}</h3>
                      <div className="card-meta-masonry">
                        <div className="meta-item-masonry">
                          <Calendar size={16} strokeWidth={2} />
                          <span>{date}</span>
                        </div>
                        {rental.review && (
                          <div className="meta-item-masonry review-masonry">
                            <MessageSquare size={16} strokeWidth={2} />
                            <span>{rental.review}</span>
                          </div>
                        )}
                      </div>
                    </div>
                  </motion.div>
                )
              })}
            </div>
          )}
        </section>

        <section className="stats-wave-section">
          <motion.div 
            className="stats-container-wave"
            initial={{ opacity: 0 }}
            whileInView={{ opacity: 1 }}
            viewport={{ once: true }}
            transition={{ duration: 0.6 }}
          >
            <div className="stat-item-wave">
              <div className="stat-icon-wave">
                <Music2 size={28} strokeWidth={2} />
              </div>
              <div className="stat-value-wave">{rentals.length}</div>
              <div className="stat-label-wave">Аренд</div>
            </div>
            <div className="stat-item-wave">
              <div className="stat-icon-wave">
                <Radio size={28} strokeWidth={2} />
              </div>
              <div className="stat-value-wave">24/7</div>
              <div className="stat-label-wave">Поддержка</div>
            </div>
            <div className="stat-item-wave">
              <div className="stat-icon-wave">
                <Volume2 size={28} strokeWidth={2} />
              </div>
              <div className="stat-value-wave">100%</div>
              <div className="stat-label-wave">Качество</div>
            </div>
          </motion.div>
        </section>
      </div>
    </div>
  )
}

export default MyRentals
